/* This script was compiled from an ESM version - do not modify directly! */
var fetcher = (function (rodash) {
	'use strict';

	const cache = {};

	class FetchError extends Error {
		constructor(message) {
			super(message);
		}
		
		get help() {
			return 'A request error occurred. Please try again later or reach out for assistance.';
		}
		
		// Shortcut for determining if bad request (400).
		isBadRequest() {
			return this instanceof BadRequestError;
		}
	}

	class RequestError extends FetchError {
		constructor(message = 'Request Error') {
			super(message);
		}
	}
	class NetworkError extends RequestError {
		constructor() {
			super('Network Error');
		}
		
		get help() {
			return 'A network error occurred. Please ensure that you are connected to the internet and try again.';
		}
	}
	class ResponseError extends FetchError {
		constructor(message, status, data) {
			super(message);
			this.status = status;
			this.data = data;
		}
		
		get help() {
			return `Reason: ${this.message}. Please try again later or reach out for assistance.`;
		}
	}
	class UnavailableError extends ResponseError {
		constructor(status, data) {
			super('Service Unavailable', status, data);
		}
		
		get help() {
			return 'The server is temporarily unavailable. Please try again later or reach out for assistance.';
		}
	}
	class ServerError extends ResponseError {
		constructor(status, data) {
			super('Server Error', status, data);
		}
		
		get help() {
			return 'An unexpected server error has occurred and has been recorded for further investigation. Please try again later or reach out for assistance.';
		}
	}
	class ClientErrorResponseError extends ResponseError {
		constructor(message, status, data) {
			super(message, status, data);
		}
		
		get help() {
			return `There was a problem with your request (reason: ${this.message}). Please try again later or reach out for assistance.`;
		}
	}
	class NotFoundError extends ClientErrorResponseError {
		constructor(status, data) {
			super('Not Found', status, data);
		}
	}
	class ForbiddenError extends ClientErrorResponseError {
		constructor(status, data) {
			super('Forbidden', status, data);
		}
	}
	class UnauthorizedError extends ClientErrorResponseError {
		constructor(status, data) {
			super('Unauthorized', status, data);
		}
	}
	class BadRequestError extends ClientErrorResponseError {
		constructor(status, data) {
			super('Bad Request', status, data);
		}
	}

	function getError(status, data) {
		switch (status) {
			case 503:
				return new UnavailableError(status, data);
			case 500:
				return new ServerError(status, data);
			case 404:
				return new NotFoundError(status, data);
			case 403:
				return new ForbiddenError(status, data);
			case 401:
				return new UnauthorizedError(status, data);
			case 400:
				return new BadRequestError(status, data);
			default:
				return new ResponseError(`Error ${status}`, status, data);
		}
	}

	class FetchData {
		get isFormData() { return this.data instanceof FormData; }

		get isEmpty() {
			if (this.isFormData) {
				return !Array.from(this.data).length;
			} else {
				return !Object.keys(this.data).length;
			}
		}

		constructor(data = {}) {
			this.data = data;
		}

		set(key, value) {
			if (this.isFormData) {
				this.data.append(key, value);
			} else {
				this.data[key] = value;
			}
		}

		toFormData() {
			if (!this.isFormData) {
				return this._objectToFormData(this.data);
			} else {
				return this.data;
			}
		}

		toJson() {
			if (this.isFormData) {
				// TODO: throw an error if data contains values that cannot be put in JSON e.g., File data.
				return JSON.stringify(this._formDataToObject(this.data));
			} else {
				return JSON.stringify(this.data);
			}
		}

		toUrlSearchParams() {
			if (this.isFormData) {
				// TODO: throw an error if data contains data that cannot be put in search params.
				return this._convertObjectToURLSearchParams(this._formDataToObject(this.data));
			} else {
				return this._convertObjectToURLSearchParams(this.data);
			}
		}

		_formDataToObject(formData) {
			const ret = {};

			for (const [key, value] of formData.entries()) {
				const existing = rodash.get(ret, key);
				if (existing) {
					if (!Array.isArray(existing)) {
						rodash.set(ret, key, [existing]);
					}
					rodash.get(ret, key).push(value);
				} else {
					rodash.set(ret, key, value);
				}
			}

			return ret;
		}

		_objectToFormData(obj, rootName) {
			const formData = new FormData();

			function appendFormData(data, root) {
				const currRoot = root || '';
				if (Array.isArray(data)) {
					for (let i = 0; i < data.length; i++) {
						appendFormData(data[i], `${currRoot}[${i}]`);
					}
				} else if (typeof data === 'object' && data) {
					Object.keys(data).forEach((key) => {
						if (currRoot === '') {
							appendFormData(data[key], key);
						} else {
							appendFormData(data[key], `${currRoot}.${key}`);
						}
					});
				} else if (data !== null && typeof data !== 'undefined') {
					formData.append(currRoot, data);
				}
			}

			appendFormData(obj, rootName);

			return formData;
		}
		
		_convertObjectToURLSearchParams(obj, prefix) {
			const params = new URLSearchParams();
			
			Object.keys(obj).forEach(key => {
				const value = obj[key];

				if (value != null) {
					let newPrefix;

					if (Array.isArray(obj)) {
						newPrefix = prefix ? `${prefix}[${key}]` : key;
					} else {
						newPrefix = prefix ? `${prefix}.${key}` : key;
					}

					if (value instanceof Array) {
						value.forEach((item, index) => {
							const newKey = `${newPrefix}[${index}]`;
							if (typeof item === 'object' && item !== null) {
								this._convertObjectToURLSearchParams(item, newKey).forEach((v, k) => params.append(k, v));
							} else {
								params.append(newKey, item);
							}
						});
					} else if (typeof value === 'object') {
						this._convertObjectToURLSearchParams(value, newPrefix).forEach((v, k) => params.append(k, v));
					} else {
						params.append(newPrefix, value);
					}
				}
			});

			return params;
		}
	}

	const fetcher = {
		_testing: {
			FetchData,
		},
		ERRORS: {
			BadRequestError,
			ClientErrorResponseError,
			FetchError,
			ForbiddenError,
			NetworkError,
			NotFoundError,
			RequestError,
			ResponseError,
			ServerError,
			UnauthorizedError,
			UnavailableError,
		},
		fetch(url, options = {}) {
			let fetchUrl = url;

			if (options.cacheKey && !options.cacheBust) {
				const cachedResult = cache[options.cacheKey];
				if (cachedResult) {
					return cachedResult;
				}
			}

			const fetchOptions = {
				method: options.form ? 'POST' : options.method,
				headers: options.headers || {},
				signal: options.controller?.signal,
			};

			const data = new FetchData(options.data);

			if (fetchOptions.method === 'POST' && window.CSRF_TOKEN && !options.excludeCsrf) {
				data.set('csrfToken', window.CSRF_TOKEN);
			}

			if (!data.isEmpty) {
				if (fetchOptions.method === 'POST') {
					if (options.form || data.isFormData) {
						fetchOptions.body = data.toFormData();
						// Ensure no set Content-Type so the browser can correctly set the boundary in the Content-Type.
						delete fetchOptions.headers?.['Content-Type'];
					} else {
						fetchOptions.headers['Content-Type'] = 'application/json';
						fetchOptions.body = data.toJson();
					}
				} else {
					const queryParams = data.toUrlSearchParams().toString();
					if (fetchUrl.includes('?')) {
						fetchUrl += `&${queryParams}`;
					} else {
						fetchUrl += `?${queryParams}`;
					}
				}
			}

			return window
				.fetch(fetchUrl, fetchOptions)
				.then(async (res) => {
					if (res.ok) {
						let result;
						if (res.headers.get('Content-Type')?.includes('json')) {
							result = {
								data: await res.json(),
								status: res.status,
							};
						} else if (res.headers.get('Content-Type')?.includes('text/html') || res.headers.get('Content-Type')?.includes('text/plain')) {
							result = {
								data: await res.text(),
								status: res.status,
							};
						} else {
							result = {
								data: {},
								status: res.status,
							};
						}

						if (options.cacheKey) {
							cache[options.cacheKey] = result;
						}

						return result;
					}

					const ret = {};
					if (res.headers.get('Content-Type')?.includes('json')) {
						await Object.assign(ret, await res.json());
					}
					
					const error = getError(res.status, ret);
					
					if (options.throwHttpError === false) {
						return {
							error,
						};
					}
					
					throw error;
				})
				.catch((e) => {
					if (e.name === 'AbortError') {
						return {
							canceled: true,
						}
					}
					
					// Firefox: "NetworkError"
					// iOS Safari: "Load failed"
					// Chrome: "Failed to fetch"
					if (e.name === 'NetworkError' || e.message.includes('NetworkError') || e.message.includes('network error') || (e.message === 'Load failed') || e.message === 'Failed to fetch') {
						throw new NetworkError();
					}
					
					throw e;
				});
		},
		
		post(url, options) {
			return fetcher.fetch(url, {
				...options,
				method: 'POST',
			});
		},
		
		fetchNoThrow(url, options) {
			return fetcher.fetch(url, {
				...options,
				throwHttpError: false,
			});
		}
	};

	return fetcher;

})(rodash);
