(() => {
	const { RentecLitElement, lit, overlayManager } = window;
	const { html, css } = lit;
	
	const SLIDE_TRANSITION_MS = 250;
	
	(class RentecDrawerComponent extends RentecLitElement {
		static properties = {
			...super.properties,
			open: { type: Boolean },
			position: { type: String, values: ['left', 'right'], default: 'left' },
			_showing: { type: Boolean, state: true },
			_panelHeight: { type: String, state: true },
		}
		
		static styles = [
			super.styles,
			css`
				:host {
					--drawer-z-index: var(--z-drawer);
				}
				
				[hidden] {
					/* Ensure drawer is not focusable when not open. */
					display: none;
				}
				
				.drawer {
					width: 100%;
					height: 100%;
					
					position: fixed;
					top: var(--drawer-top-offset, 0);
					left: 0;
					z-index: var(--drawer-z-index, 99);
				}
				
				.panel {
					min-width: var(--space-s-3);
					
					/* Ensure backdrop touch area is large enough. */
					max-width: 75vw;

					background-color: var(--color-bg-overlay);
					overflow: scroll;

					position: absolute;
					top: 0;
				}
				.panel--left {
					left: 0;
				}
				.panel--right {
					right: 0;
				}
				
				.backdrop {
					height: 100%;
					width: 100%;
					background-color: var(--color-bg-backdrop);
				}
			`,
		]
		
		render() {
			return html`
				<div class="drawer" ?hidden="${!this._showing}">
					<div id="backdrop" class="backdrop" @click="${this._onBackdropClick}"></div>
					
					<nav id="panel" class="panel panel--${this.position}" style="height: ${this._panelHeight}px;">
						<slot></slot>
					</nav>
				</div>
			`;
		}
		
		get _panelEl() { return this.shadowRoot.querySelector('#panel'); }
		get _backdropEl() { return this.shadowRoot.querySelector('#backdrop'); }
		
		disconnectedCallback() {
			super.disconnectedCallback();
			
			this._hide();
		}
		
		updated(changed) {
			super.updated(changed);
			
			if (changed.has('open')) {
				if (this.open) {
					this._show();
				} else {
					this._hide();
				}
			}
		}
		
		show() {
			this.open = true;
		}
		hide() {
			this.open = false;
		}
		
		_show() {
			if (this._showing) {
				return;
			}
			
			overlayManager.add(this, () => { this.hide(); });
			
			this._showing = true;
			
			// Manual calculation needed for iOS, where the calculation based off 100vh may not yield an accurate result.
			this._panelHeight = window.innerHeight - (parseInt(getComputedStyle(this).getPropertyValue('--drawer-top-offset')) || 0);
			
			this._panelEl.animate([
				{ translate: `${this.position === 'left' ? '-' : ''}100%` },
				{ translate: '0' },
			], { duration: SLIDE_TRANSITION_MS, easing: 'ease' });

			this._backdropEl.animate([
				{ opacity: 0 },
				{ opacity: 1 }
			], { duration: SLIDE_TRANSITION_MS, easing: 'ease' });
			
			this.dispatchEvent(new CustomEvent('show'));
		}
		
		_hide() {
			if (!this._showing) {
				return;
			}
			
			overlayManager.remove(this);
			
			Promise.all([
				this._panelEl.animate([
					{ translate: '0' },
					{ translate: `${this.position === 'left' ? '-' : ''}100%` },
				], { duration: SLIDE_TRANSITION_MS, easing: 'ease' }).finished,
				
				this._backdropEl.animate([
					{ opacity: 1 },
					{ opacity: 0 }
				], { duration: SLIDE_TRANSITION_MS, easing: 'ease' }).finished,
			]).then(() => {
				this._showing = false;
			});
			
			this.dispatchEvent(new CustomEvent('hide'));
		}
		
		_onBackdropClick () {
			this.open = false;
		}
	}).define('r-drawer');
})();
