// DEPENDENCIES: rentec-element.js

'use strict';

(() => {
	const componentName = 'r-scroll-top-btn';
	if (!customElements.get(componentName)) {
		customElements.define(
			componentName,
			/**
			 * COMPONENT: <r-scroll-top-btn>
			 * DESCRIPTION:
			 * Placing this element anywhere on a page will provide a hidden fixed-position button at the
			 * bottom-right of the screen that appears when the user scrolls down, and when clicked, scrolls the user
			 * to the top of the page.
			 *
			 * ATTRIBUTES:
			 * - Refer to the properties() method below for documentation
			 */
			class extends HTMLRentecElement {
				render() {
					return `
						<style>
							#main {
								display: none;
								opacity: 0;
								position: fixed;
								bottom: 32px;
								right: 32px;
								z-index: var(--scroll-top-btn-z-index, 99);
								background-color: var(--color-bg-scroll-top, black);
								color: white;
								cursor: pointer;
								padding: 8px;
								border-radius: 8px;
								transition: opacity 0.5s, background-color 0.5s;
								
								margin: 0;
								vertical-align: baseline;
								appearance: none;
								text-decoration: none;
								border: none;
								box-shadow: none;
							}
							
							#main:hover {
								background-color: var(--color-bg-scroll-top-hover, black);
							}
						</style>
						<div>
							<button id="main" title="Go to top" type="button" tabindex="-1">
								<svg style="display: flex;" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#FFFFFF" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M18 15l-6-6-6 6"/></svg>
							</button>
						</div>
					`;
				}
				
				constructor() {
					super();
					
					this.isShowing = false;
					
					this.ref('#main').addEventListener('click', this._onClick.bind(this));
					
					window.onscroll = (() => {
						const currentScrollFn = window.onscroll;
						return () => {
							if (currentScrollFn) {
								// avoids overwriting any existing onscroll behavior defined
								currentScrollFn.apply(this, arguments);
							}
							this.show();
						}
					})();
				}
				
				show() {
					if (document.documentElement.scrollTop > window.innerHeight) {
						if (!this.isShowing) {
							this.isShowing = true;
							this.ref('#main').style.display = 'block';
							setTimeout(() => {
								if (this.isShowing) {
									this.ref('#main').style.opacity = 0.9;
								}
							});
						}
						
					} else {
						if (this.isShowing) {
							this.isShowing = false;
							this.ref('#main').style.opacity = 0;
							setTimeout(() => {
								if (!this.isShowing) {
									this.ref('#main').style.display = 'none';
								}
							}, 500);
						}
					}
				}
				
				_onClick() {
					// Scroll smoothly to the top of the page.
					window.document.documentElement.style['scroll-behavior'] = 'smooth';
					document.documentElement.scrollTop = 0;
					setTimeout(() => {
						window.document.documentElement.style['scroll-behavior'] = null;
					});
				}
			}
		);
	}
})();
