(() => {
	const { lit, RentecFormElement, RentecLitElement } = window;
	const { html, css } = lit;
	
	class Option {
		constructor({ id, value, selected, disabled, text, group, internal = false }) {
			this.id = id;
			this.value = value;
			this.selected = selected;
			this.disabled = disabled;
			this.text = text;
			this.group = group;
			this.internal = internal;
		}
	}
	
	(class RentecSelectElement extends RentecFormElement {
		static properties = {
			...super.properties,
			
			placeholder: { type: String },
			clearable: { type: Boolean },
			loading: { type: Boolean },
			size: { type: String, values: ['md', 'sm'], default: 'md' },
			auto: { type: Boolean },
			autoOne: { type: Boolean, attribute: 'auto-one' },
			multi: { type: Boolean },
			multiSelectAll: { type: Boolean, attribute: 'multi-select-all' },
			closeOnSelect: { type: Boolean, attribute: 'close-on-select' },
			display: { type: String, values: ['text', 'chips'], default: 'text' },
			filter: { type: Boolean },
			floatingLabel: { type: Boolean, attribute: 'floating-label' },
			
			_selected: { type: Boolean, state: true },
			_focused: { type: Boolean, state: true },
			_filterInput: { type: String, state: true },
			_isOptionsVisible: { type: Boolean, state: true },
		};
		
		static styles = [
			super.styles,
			css`
				.select {
					background-image: none !important;
				}
				.select--multi {
					display: flex;
					align-items: center;
					padding-top: calc((var(--control-height-sm) - var(--space-6)) / 2 - var(--border-width-default));
					padding-bottom: calc((var(--control-height-sm) - var(--space-6)) / 2 - var(--border-width-default));
					min-height: var(--control-height-sm);
				}
				.select--multi.select--lg {
					padding-top: calc((var(--control-height-md) - var(--space-8)) / 2 - var(--border-width-default));
					padding-bottom: calc((var(--control-height-md) - var(--space-8)) / 2 - var(--border-width-default));
					min-height: var(--control-height-md);
				}

				.select--error {
					border-color: var(--color-fg-danger) !important;
				}
				.menu-button {
					display: block;
					padding: 0;
					width: 100%;
					cursor: default;
				}
				.menu-button:focus {
					outline: none;
				}
				.focused {
					border-color: var(--color-primary-light);
					outline: none;
					box-shadow: inset 0 0 0 1px var(--color-primary-light);
				}
				
				.filter {
					padding: var(--box-padding-y) var(--box-padding-x) var(--space-1);
				}
				.selection-controls {
					padding: var(--box-padding-y) var(--box-padding-x) var(--space-1);
				}
				
				.Label--floating {
					position: absolute;
					z-index: 1;
					font-size: 14px;
					line-height: 1;
					left: 16px;
					top: -8px;
					background: var(--control-color-bg);
					padding: 0 var(--space-1);
					border-radius: var(--border-radius-md);
				}

				:host(:not([slot-help]):not([help])) .help {
					display: none;
				}
			`
		];
		
		render() {
			return html`
				${this.label ? html`
					<div class="relative leading-none ${!this.floatingLabel ? 'mb-1' : ''}">
						<label for="select" class="Label Label--${this.size} ${this.floatingLabel ? 'Label--floating' : ''}">${this.label}</label>
						${this.required && !this.hideRequired ? html`
							<span style="color: var(--color-fg-danger);">*</span>
						` : ''}
					</div>
				` : ''}

				<div class="relative">
					${this.multi ? html`
						<r-menu id="menu" class="block w-full h-full" min-width="trigger" stay-open @open="${this._onMultiSelectMenuOpen}" @close="${this._onMultiSelectMenuClose}" style="--max-panel-height: var(--space-s-3);">
							<button id="menu-button" slot="trigger" class="btn menu-button" role="combobox" ?disabled="${!this._canInteract}" @focus="${this._onMultiSelectFocus}" @blur="${this._onMultiSelectBlur}">
								<div class="select select--multi ${this.size === 'md' ? `select--lg` : ''} ${!this._canInteract ? 'select--disabled' : ''} ${this.errors.length ? 'select--error' : ''} ${this._focused ? 'focused' : ''} ${this.display === 'chips' && this.value.length ? 'pl-1' : ''} relative">
									<div class="w-full" style="min-height: var(--space-6);">
										${!this.loading ? html`
											<div class="flex gap-2 w-full ${this._showClearable ? this.size === 'md' ? 'pr-8' : this.size === 'sm' ? 'pr-6' : '' : ''}">
												${this._isNoOptionsProvided
													? html`
														<div id="multitext" class="color-fg-muted">None available</div>`
													: html`
														${
															this.display === 'chips' ? html`
																${this.value.length ? html`
																	<div class="flex flex-wrap gap-1 min-w-0">
																		${this._selectedMultiOptions.map(option => html`
																			<r-chip
																				class="min-w-0"
																				removable
																				size="${this.size}"
																				._option="${option}"
																				@remove="${this.onMultiChipRemove}"
																			>
																				<div style="max-width: var(--space-s-2);">
																					<r-truncate-text text="${option.textContent}"></r-truncate-text>
																				</div>
																			</r-chip>
																		`)}
																	</div>
																` : html`
																	<div class="color-fg-muted">${this.placeholder}</div>
																`}
															` : html`
																${this.value.length ? html`
																	<div class="px-1" style="border-radius: var(--border-radius-lg); background: var(--color-bg-fill-transparent);">
																		${this.value.length}
																	</div>
																` : html`
																	<div class="color-fg-muted">${this.placeholder}</div>
																`}
																<div id="multitext" class="truncate cursor-default">${this._selectedMultiText}</div>
															`
														}
													`
												}
											</div>
										` : ''}
									</div>
									<div class="absolute top-0 right-0 h-full flex items-center color-fg-default" style="pointer-events: none;">
										<r-icon name="chevron-down"></r-icon>
									</div>
								</div>
							</button>
							${this.filter ? html`
								<div slot="header" class="filter">
									<r-input id="filter" placeholder="Search" value="${this._filterInput}" @input="${this._onFilterInput}">
										<r-icon slot="suffix" name="search"></r-icon>
									</r-input>
									${!this._isOptionsVisible ? html`
										<div class="mt-2">No options</div>
									` : ''}
								</div>
							` : ''}
							${this.multiSelectAll ? html`
								<div slot="header" class="selection-controls flex justify-end">
									<r-button variant="tertiary" size="sm" @click="${this._onSelectAllClick}">Select All</r-button>
								</div>
							` : ''}
							<slot @slotchange="${this._processOptionsSlot}" @selected="${this._onMultiOptionSelect}">
								<r-menu-item disabled>None available</r-menu-item>
							</slot>
						</r-menu>
					` : html`
						${this._isAutoOneActive ? html`
							<div data-testid="auto-one-value">${this._autoOneOption.text}</div>
						` : ''}

						<div class="relative ${this._isAutoOneActive ? 'none' : ''}">
							<select
								id="select"
								class="select ${this.size ? `select--${this.size}` : ''} ${!this._selected ? 'color-fg-muted' : ''} ${this.errors.length ? 'select--error' : ''} ${this.clearable && this._showClearable ? this.size === 'md' ? 'pr-16' : 'pr-12' : 'pr-6'}"
								autocomplete="off"
								?disabled="${!this._canInteract}"
								?required="${this.required}"
								@input="${this._onSelectElInput}"
							>
								${this._singleSelectOptionsHtml}
							</select>
							<div class="absolute top-0 right-0 h-full flex items-center color-fg-default" style="pointer-events: none;">
								<r-icon name="chevron-down"></r-icon>
							</div>
						</div>

						<slot style="display: none;" @slotchange="${this._processOptionsSlot}" @r-option-updated="${this._onOptionUpdated}"></slot>
					`}

					${this.clearable ? html`
						<div id="clear" class="absolute top-0 right-0 flex items-center justify-end h-full mr-6" style="visibility: ${this._showClearable ? 'visible' : 'hidden'}; pointer-events: none;">
							<div class="flex items-center leading-none h-full">
								<r-button variant="icon" size="${this.size}" style="pointer-events: all;" @click="${this._onClearClick}">
									<r-icon name="x"></r-icon>
								</r-button>
							</div>
						</div>
					` : ''}

					${this.loading ? html`
						<div class="absolute top-0 left-0 flex items-center h-full pl-2">
							<r-loading class="color-fg-primary"></r-loading>
						</div>
					` : ''}
				</div>

				<div class="help color-fg-muted f5 mt-1">
					<slot name="help">
						<div>${this.help}</div>
					</slot>
				</div>

				${this.errors?.map(e => html`
					<r-form-error>${e}</r-form-error>
				`)}
			`;
		}
		
		get _selectEl() { return this.shadowRoot.querySelector('#select'); }
		get _clearEl() { return this.shadowRoot.querySelector('#clear'); }
		get _multitextEl() { return this.shadowRoot.querySelector('#multitext'); }
		get _menuEl() { return this.shadowRoot.querySelector('#menu'); }
		get _menuButtonEl() { return this.shadowRoot.querySelector('#menu-button'); }
		get _filterEl() { return this.shadowRoot.querySelector('#filter'); }
		
		get _selectElSelectedOption() { return this._selectEl.options[this._selectEl.selectedIndex]; }
		
		get _isNoOptionsProvided() { return !this.getAllOptions().length; }
		
		get _showClearable() { return this._canInteract && (this.multi ? !!this.value?.length : (this.clearable && !this.auto && !this._isAutoOneActive && !!this._singleSelectOptions.find(option => option.selected) && this.value !== '')); }
		
		get _selectedMultiText() { return this._selectedMultiOptions.map(option => option.textContent).join(', '); }
		get _selectedMultiOptions() { return Array.isArray(this.value) ? this.value.map(value => this.getAllOptions().find(option => option.value === value)).filter(option => !!option) : []; }
		
		get _canInteract() { return !this.disabled && !this.loading; }
		
		get submittable() { return !this.disabled && !this.loading; }
		get visibleValue() { return this.multi ? this._selectedMultiText : this._singleSelectOptions.find(option => option.selected && option.value === this.value)?.text; }
		
		get _singleSelectOptions() {
			const options = [];
			
			const value = this.value;
			
			if (this.loading) {
				options.push(new Option({
					id: 'loading',
					value: '',
					disabled: true,
					selected: true,
					text: '',
					internal: true,
				}));
				
				return options;
			}
			
			if (this._isNoOptionsProvided) {
				options.push(new Option({
					id: 'noneAvailable',
					value: '',
					disabled: true,
					selected: true,
					text: 'None available',
					internal: true,
				}));
			} else {
				this.getAllOptions().forEach(el => {
					const optionValue = el.value
					options.push(new Option({
						value: optionValue,
						disabled: el.disabled,
						selected: optionValue === value,
						text: el.textContent,
						id: el,
						el: el,
						group: el.parentElement.nodeName === 'R-OPTGROUP' ? el.parentElement.label : null,
					}));
				});
				
				const addClearOption = this.clearable && !this.autoOne && !this['auto'] && options[0]?.value !== '';
				if (addClearOption) {
					options.unshift(new Option({
						id: 'clear',
						value: '',
						selected: value === '' && !this.placeholder,
						text: '',
						disabled: false,
						internal: true,
					}));
				}
				
				if (this.placeholder) {
					options.unshift(new Option({
						id: 'placeholder',
						value: '',
						selected: value === '',
						text: `${this.placeholder}`,
						disabled: true,
						internal: true,
					}));
				}
				
				// Handle empty string option.
				if (options[0].selected && options.slice(1).find(o => o.selected)) {
					options[0].selected = false;
				}
			}
			
			return options;
		}
		
		get _autoOneOption() {
			const options = this._singleSelectOptions.filter(o => !o.internal && !o.disabled);
			if (options.length === 1) {
				return options[0];
			}
			return null;
		}
		get _isAutoOneOptionAvailable() {
			return !!this._autoOneOption;
		}
		get _isAutoOneActive() {
			return this.autoOne && !this.loading && !this.disabled && this._isAutoOneOptionAvailable;
		}
		get _singleSelectOptionsHtml() {
			const options = [];
			let group = null;
			let groupOptions = [];
			
			const addGroup = (label) => {
				options.push(html`
					<optgroup label="${label}">
						${groupOptions}
					</optgroup>
				`);
				group = null;
				groupOptions = [];
			}
			
			this._singleSelectOptions.forEach((option, i) => {
				if (option.group) {
					if (group) {
						if (this._singleSelectOptions[i - 1].group !== option.group) {
							addGroup(group);
						}
					}
					
					groupOptions.push(html`
						<option ?disabled="${option.disabled}" .value="${option.value}" .selected="${option.selected}">${option.text}</option>
					`);
					
					group = option.group;
				} else {
					if (group) {
						addGroup(group)
					}
					
					options.push(html`
						<option ?disabled="${option.disabled}" .value="${option.value}" .selected="${option.selected}">${option.text}</option>
					`);
				}
			});
			
			if (group) {
				addGroup(group);
			}
			
			return options;
		}
		
		clear() {
			if (this.multi) {
				this.value = [];
				this._emitUpdated();
			} else {
				this.value = '';
				this._emitUpdated();
			}
		}
		
		getAllOptions() {
			return Array.from(this.querySelectorAll('r-option'));
		}
		
		_onClearClick() {
			this.clear();
		}
		
		_onSelectElInput(e) {
			e.stopImmediatePropagation();
			
			this.value = this._singleSelectOptions[this._selectEl.selectedIndex].value;
			this._emitUpdated();
		}
		
		_processOptionsSlot() {
			if (this.ready) {
				this._updateSlottedOptions();
			} else {
				this.onReady.then(() => {
					this._updateSlottedOptions();
				});
			}
		}
		
		_updateSlottedOptions() {
			// Validate options
			let invalid;
			
			this.getAllOptions().forEach(el => {
				if (el.value == null) {
					invalid = true;
				}
			});
			
			setTimeout(() => {
				if (invalid) {
					throw new Error('Warning: invalid option values exist.');
				}
			});
			
			if (this.multi) {
				this.getAllOptions().forEach(option => {
					option._multi = true;
					
					// If an option is slotted in where its value is found in the value, it should be selected.
					if (this.value.includes(option.value)) {
						option.selected = true;
					}
				});
				
				this.setValueToMultiSelectedOptions();
			} else {
				this.requestUpdate();
			}
		}
		
		_onMultiOptionSelect(e) {
			e.stopPropagation();
			this.setValueToMultiSelectedOptions();
			
			if (this.closeOnSelect) {
				this._menuEl.close();
			}
			
			if (this._menuEl.show) {
				this._focused = true;
			}
			
			this._emitUpdated();
		}
		
		willUpdate(changed) {
			super.willUpdate(changed);
			
			if (this.multi) {
				this._selected = this.getAllOptions().filter(option => option.selected).length;
			} else {
				if (this._isNoOptionsProvided) {
					this._selected = false;
				} else {
					this._selected =  !!this._singleSelectOptions.find(o => {
						if (this.placeholder && o.id === 'placeholder') {
							return false;
						}
						return o.selected;
					});
				}
			}
		}
		
		firstUpdated() {
			this._firstUpdate = true;
		}
		
		setValueToMultiSelectedOptions() {
			if (this.loading) {
				// Options should not update the value when loading.
				return;
			}
			
			if (!Array.isArray(this.value)) {
				this.value = [];
			}
			
			const selectedOptions = this.getAllOptions().filter(el => el.selected === true);
			const selectedOptionValues = new Set(selectedOptions.map(el => el.value));
			const existingValues = new Set(this.value);
			
			// value: [1,2]. selected options: [2]
			const valueNotSelectedOption = !!this.value.filter(value => !selectedOptionValues.has(value)).length;
			// value: [1,2]. selected options: [1,2,3]
			const selectedOptionNotInValue = !!selectedOptions.filter(option => !existingValues.has(option.value)).length;
			
			const valueNeedsUpdate = valueNotSelectedOption || selectedOptionNotInValue;
			if (valueNeedsUpdate) {
				this.value = selectedOptions.map(option => option.value);
			}
		}
		
		updated(changed) {
			super.updated(changed);
			
			this.dispatchEvent(new CustomEvent('r-select-updated'));
			
			if (this.ready && !this.loading) {
				if (this.multi) {
					if (!this.clearable) {
						this.clearable = true;
					}
					
					// Value should be an array.
					if (!Array.isArray(this.value)) {
						this.value = [];
					}
					
					if (changed.has('value') && !this._firstUpdate) {
						// Unselect any option where the option value is not in the value.
						this.getAllOptions().forEach(el => {
							if (el.selected && !this.value.includes(el.value)) {
								el.selected = false;
							}
						});
					}
					
					// 1. Select any options that are in the value.
					this.getAllOptions().forEach(el => {
						if (this.value.includes(el.value) && !el.selected) {
							el.selected = true;
						}
					});
					
					// 2. Set value as the selected options.
					this.setValueToMultiSelectedOptions();
					
					// 3. Remove any value in the value that is not an option.
					const invalidValues = this.value.filter(value => !this.getAllOptions().find(el => el.value === value)).length;
					if (invalidValues) {
						this.value = this.value.filter(value => this.getAllOptions().find(option => option.value === value));
					}
					
				} else {
					if (this.value == null) {
						this.value = '';
					}
					
					if (!this._singleSelectOptions.find(o => o.selected)) {
						// No internal option is selected
						
						if (this.placeholder) {
							// ... so change the selected value to the placeholder.
							this._selectEl.selectedIndex = 0;
						} else {
							this._selectEl.selectedIndex = -1;
						}
						
						// ... clear out any set value.
						if (this.value !== '') {
							this.value = '';
						}
					} else {
						this._selectEl.selectedIndex = this._singleSelectOptions.findIndex(o => o.selected);
					}
					
					if (this._isNoOptionsProvided) {
						if (this.value !== '') {
							this.value = '';
						}
					} else {
						if (this.auto && this.value === '') {
							const optionToAutoSelect = this._singleSelectOptions.find(o => !o.disabled && !o.internal);
							if (optionToAutoSelect) {
								this.value = optionToAutoSelect.value;
								this._emitUpdated();
							}
						}
						if (this.autoOne && this._isAutoOneActive && this.value === '' && this._isAutoOneOptionAvailable) {
							this.value = this._autoOneOption.value;
							this._emitUpdated();
						}
					}
				}
				
				this._updateFilterVisibleOptions();
			}
			
			this._firstUpdate = false;
		}
		
		_emitUpdated() {
			this.dispatchEvent(new CustomEvent('change', { bubbles: true }));
			this.dispatchEvent(new CustomEvent('input', { bubbles: true }));
		}
		
		checkValidity() {
			return this._checkRequiredValuePresent();
		}
		reportValidity() {
			return this._reportValidity(() => this._reportRequiredValuePresent());
		}
		
		focus() {
			if (this.multi) {
				this._menuButtonEl.focus();
			} else {
				this._selectEl.focus();
			}
		}
		
		_onMultiSelectFocus() {
			this._focused = true;
		}
		_onMultiSelectBlur() {
			setTimeout(() => {
				// Retain focus state if focused on one of the options.
				if (!this.getAllOptions().includes(document.activeElement)) {
					this._focused = false;
				}
			});
		}
		_onMultiSelectMenuOpen() {
			if (this.filter) {
				this._filterEl.focus();
			}
		}
		_onMultiSelectMenuClose() {
			if (this.getAllOptions().includes(document.activeElement) || document.activeElement === this) {
				// When the menu is closed, focus the menu trigger if the active element is one of the menu options.
				this.focus();
			} else {
				this._focused = false;
			}
			
			if (this.filter) {
				this._filterInput = '';
			}
		}
		onMultiChipRemove(e) {
			this.value = this.value.filter(i => i !== e.target._option.value);
			this._emitUpdated();
		}
		
		_onFilterInput() {
			this._filterInput = this._filterEl.value;
		}
		_updateFilterVisibleOptions() {
			this.getAllOptions().forEach(el => {
				el.hidden = this._filterInput && !el.textContent?.toLowerCase().includes(this._filterInput.toLowerCase());
			});
			this._isOptionsVisible = this.getAllOptions().find(el => !el.hidden);
		}
		
		_onSelectAllClick() {
			this.value = this.getAllOptions().map(el => el.value);
			this._emitUpdated();
		}
		
		_onOptionUpdated() {
			if (!this.multi) {
				// Sync internal native select options.
				this.requestUpdate();
			}
		}
	}).define('r-select');
	
	(class RentecOptionComponent extends RentecLitElement {
		static properties = {
			...super.properties,
			disabled: { type: Boolean },
			selected: { type: Boolean },
			value: {},
			hidden: { type: Boolean },
			
			_multi: { type: Boolean, state: true },
		}
		
		static styles = [
			super.styles,
			css`
				:host {
					display: contents;
				}
			`
		]
		
		get value() {
			if (this._value != null) {
				return this._value;
			}
			return this.textContent;
		}
		set value(val) {
			this._value = val;
			this.requestUpdate();
		}
		
		updated(changed) {
			super.updated(changed);
			this.dispatchEvent(new CustomEvent('r-option-updated', { bubbles: true }));
		}
		
		render() {
			if (this._multi) {
				return html`
					<r-menu-item .checked="${this.selected}" ?hidden="${this.hidden}" type="check" @click="${this._onClick}">
						<div class="text-left">
							<slot></slot>
						</div>
					</r-menu-item>
				`
			}
			return null;
		}
		
		toggle() {
			this.selected = !this.selected;
			this.dispatchEvent(new CustomEvent('selected', {
				bubbles: true,
			}));
		}
		
		_onClick() {
			this.toggle();
		}
	}).define('r-option');
	
	(class RentecOptgroupComponent extends RentecLitElement {
		static properties = {
			...super.properties,
			label: { type: String, required: true },
			_multi: { type: Boolean, state: true },
		}
		
		static styles = [
			super.styles,
			css`
				:host(:not(:last-child)) {
					border-bottom: 1px solid var(--color-border-muted);
				}
				.optgroup {
					padding: var(--space-2) var(--space-2);
				}
				.optgroup_label {
					margin-left: var(--space-2);
					font-size: var(--fs-f5);
					line-height: var(--fs-f5-lh);
				}
			`
		]
		
		render() {
			if (this._multi) {
				return html`
					<div class="optgroup">
						<div class="optgroup_label">${this.label}</div>
					</div>
					<slot></slot>
				`;
			} else {
				return html`<slot></slot>`;
			}
		}
		
		connectedCallback() {
			super.connectedCallback();
			
			this._select = this.closest('r-select');
			this._multi = this._select.multi;
			this._onSelectUpdated = () => {
				this._multi = this._select.multi;
			}
			this._select.addEventListener('r-select-updated', this._onSelectUpdated);
		}
		disconnectedCallback() {
			super.disconnectedCallback();
			this._select.removeEventListener('r-select-updated', this._onSelectUpdated);
		}
	}).define('r-optgroup');
})();
