// DEPENDENCIES: rentec-css-utils.js

'use strict';

/**
 * Class to extend off of when creating web components (custom elements)
 * Features:
 * - Enables shadow-dom on the Element
 * - Maps attributes to variables defined on the element
 * - Assigns passed stylesheets
 * - Exposes some convenience methods
 * Usage:
 * - Methods available to override: render(), properties(), stylesheets()
 * 		- render() MUST be overridden
 */
window.HTMLRentecElement = class HTMLRentecElement extends HTMLElement {
	// Note: must not inspect element attributes during the constructor()
	// https://html.spec.whatwg.org/multipage/custom-elements.html#custom-element-conformance
	constructor() {
		super();
		
		this.attachShadow({ mode: "open" });
		
		this.properties().forEach((prop) => {
			Object.defineProperty(this, prop, {
				get: () => {
					return this.getAttribute(prop);
				},
				set: (value) => {
					if (value === null) {
						this.removeAttribute(prop);
					} else {
						this.setAttribute(prop, value);
					}
				},
			})
		});
		
		this._addTemplate();
	}
	
	/**
	 * Hook to perform any actions before attaching the element to the root
	 */
	beforeRender() {}
	
	/**
	 * Responsible for returning the string template used for constructing the template element.
	 * Every extending class MUST implement this method.
	 */
	render() {
		throw new Error('Must override render() method');
	}
	
	/**
	 * Returns the array of attributes to map to variables.
	 * If defining attributes that the element accepts, one should specify them here.
	 * Setting these variables reflects them back onto the element.
	 * Setting these variables to null will remove those attributes from the element.
	 * @returns {String[]}
	 */
	properties() {
		return [];
	}
	
	/**
	 * Returns an array of hrefs that correspond to the CSS stylesheet to assign to this element.
	 * @returns {String[]}
	 */
	stylesheets() {
		return [];
	}
	
	/**
	 * Convenience helper to get an element.
	 * @param str - element DOM selector
	 * @returns {HTMLElement}
	 */
	ref(str) {
		return this.shadowRoot.querySelector(str);
	}
	
	/**
	 * Convenience helper to get all matching elements.
	 * @param str - element DOM selector
	 * @returns {HTMLElement}
	 */
	refAll(str) {
		return this.shadowRoot.querySelectorAll(str);
	}
	
	/**
	 * Convenience helper to return the children of an element as an array
	 * @param str
	 * @returns {Element[]}
	 */
	refChildren(str) {
		return Array.from(this.ref(str).children);
	}
	
	/**
	 * Attach stylesheets to this custom element.
	 * @private
	 */
	_attachStyleSheets() {
		this.stylesheets().forEach((href) => {
			const stylesheet = document.createElement('link');
			stylesheet.href = href;
			stylesheet.rel = 'stylesheet';
			this.shadowRoot.appendChild(stylesheet);
		});
	}
	
	/**
	 * Attach the template to this custom element.
	 * @private
	 */
	_addTemplate() {
		this.beforeRender();
		
		this._attachStyleSheets();
		
		const template = document.createElement("template");
		template.innerHTML = this.render();
		this.shadowRoot.appendChild(template.content.cloneNode(true));
	}
}
