window.themeManager = {
	themes: {
		SYSTEM: 'system',
		LIGHT: 'light',
		DARK: 'dark',
		DARK_HIGH_CONTRAST: 'dark-high-contrast',
	},
	
	init: function ({ isLoggedIn = false } = {}) {
		this._isLoggedIn = isLoggedIn;
		
		this._metaLightThemeColor =  document.querySelector('[media="(prefers-color-scheme: light)"]')?.content || '#fff';
		this._metaDarkThemeColor =  document.querySelector('[media="(prefers-color-scheme: dark)"]')?.content || '#000';
		
		this.update();
		
		// Listen for system changes in real-time
		window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', e => {
			this.update();
		});
		
		document.addEventListener('visibilitychange', () => {
			if (document.visibilityState === 'visible') {
				this.update();
			}
		});
	},
	
	update: function() {
		const systemDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
		
		// Local Storage > System Preference
		if (this.theme === this.themes.DARK || (this.theme === this.themes.SYSTEM && systemDark)) {
			document.documentElement.setAttribute('data-theme', this.themes.DARK);
			this._updateMetaTag(this.themes.DARK);
		} else if (this.theme === this.themes.DARK_HIGH_CONTRAST) {
			document.documentElement.setAttribute('data-theme', this.themes.DARK_HIGH_CONTRAST);
			this._updateMetaTag(this.themes.DARK_HIGH_CONTRAST);
		} else {
			document.documentElement.setAttribute('data-theme', this.themes.LIGHT);
			this._updateMetaTag(this.themes.LIGHT);
		}
		
		this._syncToServer();
	},
	async _syncToServer() {
		if (!this._isLoggedIn) { return; }
		
		const syncedTheme = window.localStorage.getItem('synced-theme');
		const activeTheme = document.documentElement.getAttribute('data-theme');
		
		if (activeTheme === this._syncingTheme) { return; }
		
		if (!syncedTheme || syncedTheme !== activeTheme) {
			this._syncingTheme = activeTheme;
			
			try {
				const res = await fetch('SaveWebTheme.action', {
					method: 'POST',
					headers: {
						'Content-Type': 'application/json',
					},
					body: JSON.stringify({
						theme: activeTheme,
						csrfToken: window.CSRF_TOKEN,
					}),
				});
				if (res.ok && res.status === 200) {
					window.localStorage.setItem('synced-theme', activeTheme);
				}
			} catch (e) {
				console.error(e);
			} finally {
				this._syncingTheme = null;
			}
		}
	},
	
	get theme() {
		const localTheme = window.localStorage.getItem('theme');
		
		if (![this.themes.SYSTEM, this.themes.LIGHT, this.themes.DARK, this.themes.DARK_HIGH_CONTRAST].includes(localTheme)) {
			return this.themes.SYSTEM;
		}
		
		return localTheme;
	},
	set theme(theme) {
		window.localStorage.setItem('theme', theme);
		this.update();
	},
	
	_updateMetaTag(theme) {
		const metaThemeColors = document.querySelectorAll('meta[name="theme-color"]');
		
		if (!metaThemeColors.length) {
			const metaEl = document.createElement('meta');
			metaEl.name = 'theme-color';
			document.head.appendChild(metaEl);
			metaThemeColors.push(metaEl);
		}
		
		let color;
		
		if (theme === this.themes.DARK || theme === this.themes.DARK_HIGH_CONTRAST) {
			color = this._metaDarkThemeColor;
		} else if (theme === this.themes.LIGHT) {
			color = this._metaLightThemeColor;
		}
		
		metaThemeColors.forEach(el => el.setAttribute('content', color));
	}
};